#include <cmath>
#include <algorithm>
#include "jacobi.h"

using namespace std;


#define ABS(x) ( (x)>0 ? (x) : (-(x)) )
#define SGN(x) ( (x)<0 ? -1 : 1 )
#define SQRT sqrt
#define POWER pow

real symmetric_matrix::factor = POWER(10.0,12);


symmetric_matrix::symmetric_matrix(int ord, real scalar)
{
	int i, j;

	this->ord = ord;

	a = new real*[ord];
	a[0] = new real[ord*ord];

	for (i=1; i<ord; i++)
		a[i] = a[i-1] + ord;

	for (i=0; i<ord; i++)
		for (j=0; j<ord; j++)
			a[i][j] = 0;

	for (i=0; i<ord; i++)
		a[i][i] = scalar;
}

symmetric_matrix::symmetric_matrix(const symmetric_matrix &m)
{
	int i, j;

	ord = m.ord;
	a = new real*[ord];
	a[0] = new real[ord*ord];

	for (i=1; i<ord; i++)
		a[i] = a[i-1] + ord;

	for (i=0; i<ord; i++)
		for (j=0; j<ord; j++)
			a[i][j] = m.a[i][j];
}

symmetric_matrix::~symmetric_matrix()
{
	delete [] a[0];
	delete [] a;
}

symmetric_matrix& symmetric_matrix::operator=(const symmetric_matrix &m)
{
	int i, j;

	if (&m==this)
		return *this;

	ord = m.ord;
	delete [] a[0];
	delete [] a;
	a = new real*[ord];
	a[0] = new real[ord*ord];

	for (i=1; i<ord; i++)
		a[i] = a[i-1] + ord;

	for (i=0; i<ord; i++)
		for (j=0; j<ord; j++)
			a[i][j] = m.a[i][j];

	return *this;
}

bool symmetric_matrix::operator==(const symmetric_matrix &m) const
{
	int i, j;

	if (ord!=m.ord)
		return false;

	for (i=0; i<ord; i++)
		for (j=0; j<ord; j++)
			if (a[i][j]!=m.a[i][j])
				return false;

	return true;
}

bool symmetric_matrix::operator!=(const symmetric_matrix &m) const
{
	return !operator==(m);
}

int symmetric_matrix::order() const
{
	return ord;
}

real& symmetric_matrix::operator()(int row, int column)
{
	if (column<row)
	{
		int temp = row;
		row = column;
		column = temp;
	}

	return a[--row][--column];
}

real symmetric_matrix::norm2() const
{
	real result = 0;
	int i, j;

	for (i=0; i<ord; i++)
		for (j=i+1; j<ord; j++)
			result += a[i][j]*a[i][j];

	return result;
}

vector<real> symmetric_matrix::spectrum() const
{
    vector<real> v(ord);

    if (ord==0)
        return v;

    if (ord==1)
    {
        v[0] = floor(a[0][0]*factor+0.5)/factor;

        return v;
    }

	symmetric_matrix m(*this);
	real current_norm = norm2();

	while (true)
	{
		real max = -1;
		int k, l, kcur, lcur;

		for (kcur=0; kcur<ord; kcur++)
			for (lcur=kcur+1; lcur<ord; lcur++)
			{
				real n = ABS(m.a[kcur][lcur]);

				if (n>max)
				{
					max = n;
					k = kcur;
					l = lcur;
				}
			}

		k++;
		l++;

		real lambda = -m(k,l), mu = (m(l,l)-m(k,k))/2;
		real omega = SGN(mu)*lambda/SQRT(lambda*lambda+mu*mu);
		real s = omega/SQRT(2*(1+SQRT(1-omega*omega)));
		real c = SQRT(1-s*s);

		int i, j;
		symmetric_matrix copy(m);

		for (j=1; j<=ord; j++)
			if (j!=k && j!=l)
				m(k,j) = copy(k,j)*c+copy(l,j)*s;

		for (j=1; j<=ord; j++)
			if (j!=k && j!=l)
				m(l,j) = copy(l,j)*c-copy(k,j)*s;

		for (i=1; i<=ord; i++)
			if (i!=k && i!=l)
				m(i,k) = copy(i,k)*c+copy(i,l)*s;

		for (i=1; i<=ord; i++)
			if (i!=k && i!=l)
				m(i,l) = copy(i,l)*c-copy(i,k)*s;

		m(k,k) = copy(k,k)*c*c+copy(l,l)*s*s+2*copy(k,l)*s*c;
		m(l,l) = copy(k,k)*s*s+copy(l,l)*c*c-2*copy(k,l)*s*c;
		m(k,l) = (copy(l,l)-copy(k,k))*s*c+copy(k,l)*(c*c-s*s);
		real new_norm = m.norm2();

		if (new_norm>=current_norm || current_norm<1e-20)
			break;

		current_norm = new_norm;
	}

	int i;

	for (i=0; i<ord; i++)
		v[i] = floor(m.a[i][i]*factor+0.5)/factor;

	stable_sort(v.begin(), v.end());

	return v;
}

ostream& operator<<(ostream &ostr, symmetric_matrix &m)
{
	int i, j;

	for (i=1; i<=m.ord; i++)
	{
		for (j=1; j<=m.ord; j++)
			ostr << m(i,j) << "\t";

		ostr << "\n";
	}

	return ostr;
}
